<?php

namespace app\controllers;

use Yii;
use yii\web\Controller;
use yii\web\NotFoundHttpException;
use yii\filters\AccessControl;
use yii\data\ActiveDataProvider;
use app\models\Application;
use app\models\Tour;
use app\models\User;
use yii\web\UploadedFile;

class AdminController extends Controller
{
    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    [
                        'allow' => true,
                        'roles' => ['@'],
                        'matchCallback' => function ($rule, $action) {
                            return Yii::$app->user->identity->is_admin == 1;
                        },
                    ],
                ],
            ],
        ];
    }

    /**
     * Главная страница админки
     */
    public function actionIndex()
    {
        return $this->render('index');
    }

    /**
     * Список заявок с фильтрацией и сортировкой
     */
    public function actionApplications()
    {
        $searchModel = new Application();
        $query = Application::find()->with(['tour', 'user']);

        // Фильтрация по статусу
        $status = Yii::$app->request->get('status');
        if ($status !== null && $status !== '') {
            $query->andWhere(['status' => $status]);
        }

        // Фильтрация по туру
        $tour_id = Yii::$app->request->get('tour_id');
        if ($tour_id !== null && $tour_id !== '') {
            $query->andWhere(['tour_id' => $tour_id]);
        }

        // Сортировка
        $sort = Yii::$app->request->get('sort', '-id');
        if ($sort == 'id') {
            $query->orderBy('id ASC');
        } elseif ($sort == '-id') {
            $query->orderBy('id DESC');
        } elseif ($sort == 'date_start') {
            $query->orderBy('date_start ASC');
        } elseif ($sort == '-date_start') {
            $query->orderBy('date_start DESC');
        }

        $dataProvider = new ActiveDataProvider([
            'query' => $query,
            'pagination' => [
                'pageSize' => 20,
            ],
        ]);

        return $this->render('applications', [
            'dataProvider' => $dataProvider,
            'searchModel' => $searchModel,
            'tours' => Tour::find()->all(),
        ]);
    }

    /**
     * Просмотр и изменение статуса заявки
     */
    public function actionViewApplication($id)
    {
        $model = $this->findApplicationModel($id);

        if (Yii::$app->request->isPost) {
            $newStatus = Yii::$app->request->post('status');
            if (in_array($newStatus, array_keys(Application::getStatuses()))) {
                $model->status = $newStatus;
                if ($model->save()) {
                    Yii::$app->session->setFlash('success', 'Статус заявки обновлен.');
                } else {
                    Yii::$app->session->setFlash('error', 'Ошибка при обновлении статуса.');
                }
            }
        }

        return $this->render('view-application', [
            'model' => $model,
        ]);
    }

    /**
     * Управление турами - список
     */
    public function actionTours()
    {
        $dataProvider = new ActiveDataProvider([
            'query' => Tour::find(),
            'pagination' => [
                'pageSize' => 20,
            ],
        ]);

        return $this->render('tours', [
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Создание нового тура
     */
    public function actionCreateTour()
    {
        $model = new Tour();

        if ($model->load(Yii::$app->request->post())) {
            $model->imageFile = UploadedFile::getInstance($model, 'imageFile');

            if ($model->validate()) {
                if ($model->imageFile) {
                    // Генерируем уникальное имя файла
                    $fileName = 'tour_' . time() . '_' . Yii::$app->security->generateRandomString(8) . '.' . $model->imageFile->extension;
                    $model->image = $fileName;

                    // Сохраняем в папку web/images
                    $imagePath = Yii::getAlias('@webroot/images/') . $fileName;
                    $model->imageFile->saveAs($imagePath);
                }

                if ($model->save(false)) {
                    Yii::$app->session->setFlash('success', 'Тур успешно создан.');
                    return $this->redirect(['tours']);
                }
            }
        }

        return $this->render('tour-form', [
            'model' => $model,
        ]);
    }

    /**
     * Редактирование тура
     */
    public function actionUpdateTour($id)
    {
        $model = $this->findTourModel($id);
        $oldImage = $model->image;

        if ($model->load(Yii::$app->request->post())) {
            $model->imageFile = UploadedFile::getInstance($model, 'imageFile');

            if ($model->validate()) {
                if ($model->imageFile) {
                    // Генерируем уникальное имя файла
                    $fileName = 'tour_' . time() . '_' . Yii::$app->security->generateRandomString(8) . '.' . $model->imageFile->extension;
                    $model->image = $fileName;

                    // Сохраняем в папку web/images
                    $imagePath = Yii::getAlias('@webroot/images/') . $fileName;
                    $model->imageFile->saveAs($imagePath);

                    // Удаляем старую картинку, если она существует
                    if ($oldImage && file_exists(Yii::getAlias('@webroot/images/') . $oldImage)) {
                        unlink(Yii::getAlias('@webroot/images/') . $oldImage);
                    }
                } else {
                    $model->image = $oldImage;
                }

                if ($model->save(false)) {
                    Yii::$app->session->setFlash('success', 'Тур успешно обновлен.');
                    return $this->redirect(['tours']);
                }
            }
        }

        return $this->render('tour-form', [
            'model' => $model,
        ]);
    }

    /**
     * Удаление тура
     */
    public function actionDeleteTour($id)
    {
        $model = $this->findTourModel($id);

        // Удаляем картинку
        if ($model->image && file_exists(Yii::getAlias('@webroot/images/') . $model->image)) {
            unlink(Yii::getAlias('@webroot/images/') . $model->image);
        }

        $model->delete();

        Yii::$app->session->setFlash('success', 'Тур успешно удален.');
        return $this->redirect(['tours']);
    }

    /**
     * Управление пользователями
     */
    public function actionUsers()
    {
        $dataProvider = new ActiveDataProvider([
            'query' => User::find(),
            'pagination' => [
                'pageSize' => 20,
            ],
        ]);

        return $this->render('users', [
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Редактирование пользователя
     */
    public function actionUpdateUser($id)
    {
        $model = $this->findUserModel($id);

        if ($model->load(Yii::$app->request->post())) {
            if ($model->save()) {
                Yii::$app->session->setFlash('success', 'Пользователь успешно обновлен.');
                return $this->redirect(['users']);
            }
        }

        return $this->render('user-form', [
            'model' => $model,
        ]);
    }

    /**
     * Удаление пользователя
     */
    public function actionDeleteUser($id)
    {
        $model = $this->findUserModel($id);
        $model->delete();

        Yii::$app->session->setFlash('success', 'Пользователь успешно удален.');
        return $this->redirect(['users']);
    }

    /**
     * Вспомогательные методы для поиска моделей
     */
    protected function findApplicationModel($id)
    {
        $model = Application::findOne($id);
        if ($model === null) {
            throw new NotFoundHttpException('Заявка не найдена.');
        }
        return $model;
    }

    protected function findTourModel($id)
    {
        $model = Tour::findOne($id);
        if ($model === null) {
            throw new NotFoundHttpException('Тур не найден.');
        }
        return $model;
    }

    protected function findUserModel($id)
    {
        $model = User::findOne($id);
        if ($model === null) {
            throw new NotFoundHttpException('Пользователь не найден.');
        }
        return $model;
    }
}